import path from "path";
import "dotenv/config";
import * as express from "express";
import express__default from "express";
import cors from "cors";
import initSqlJs from "sql.js";
import fs from "fs";
import { fileURLToPath } from "url";
const handleDemo = (req, res) => {
  const response = {
    message: "Hello from Express server"
  };
  res.status(200).json(response);
};
const __dirname$2 = path.dirname(fileURLToPath(import.meta.url));
const dbPath = path.join(__dirname$2, "..", "applications.db");
let dbInstance = null;
async function initDb() {
  if (dbInstance) return dbInstance;
  const SQL = await initSqlJs();
  let db;
  if (fs.existsSync(dbPath)) {
    const data = fs.readFileSync(dbPath);
    db = new SQL.Database(data);
  } else {
    db = new SQL.Database();
  }
  dbInstance = db;
  return db;
}
function saveDb() {
  if (dbInstance) {
    const data = dbInstance.export();
    const buffer = Buffer.from(data);
    fs.writeFileSync(dbPath, buffer);
  }
}
async function initializeDatabase() {
  const database = await initDb();
  database.run(`
    CREATE TABLE IF NOT EXISTS applications (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      firstName TEXT NOT NULL,
      lastName TEXT NOT NULL,
      email TEXT NOT NULL,
      phone TEXT NOT NULL,
      ssn TEXT NOT NULL,
      idType TEXT NOT NULL,
      address TEXT NOT NULL,
      city TEXT NOT NULL,
      state TEXT NOT NULL,
      zipCode TEXT NOT NULL,
      bankName TEXT NOT NULL,
      accountType TEXT NOT NULL,
      routingNumber TEXT NOT NULL,
      accountNumber TEXT NOT NULL,
      status TEXT DEFAULT 'pending',
      submittedAt DATETIME DEFAULT CURRENT_TIMESTAMP,
      reviewedAt DATETIME,
      reviewedBy TEXT,
      notes TEXT
    );

    CREATE TABLE IF NOT EXISTS admin_users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      email TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL,
      name TEXT NOT NULL,
      createdAt DATETIME DEFAULT CURRENT_TIMESTAMP
    );
  `);
  saveDb();
  const stmt = database.prepare("SELECT COUNT(*) as count FROM admin_users WHERE email = ?");
  stmt.bind(["admin@empowerment.org"]);
  stmt.step();
  const result = stmt.getAsObject();
  stmt.free();
  if (result.count === 0) {
    const hashedPassword = Buffer.from("admin123").toString("base64");
    const insertStmt = database.prepare(
      "INSERT INTO admin_users (email, password, name) VALUES (?, ?, ?)"
    );
    insertStmt.bind(["admin@empowerment.org", hashedPassword, "Administrator"]);
    insertStmt.step();
    insertStmt.free();
    saveDb();
  }
  console.log("Database initialized successfully");
}
async function submitApplication(data) {
  const database = await initDb();
  const stmt = database.prepare(`
    INSERT INTO applications (
      firstName, lastName, email, phone, ssn, idType,
      address, city, state, zipCode, bankName, accountType,
      routingNumber, accountNumber
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
  `);
  stmt.bind([
    data.firstName,
    data.lastName,
    data.email,
    data.phone,
    data.ssn,
    data.idType,
    data.address,
    data.city,
    data.state,
    data.zipCode,
    data.bankName,
    data.accountType,
    data.routingNumber,
    data.accountNumber
  ]);
  stmt.step();
  stmt.free();
  saveDb();
  const idStmt = database.prepare("SELECT last_insert_rowid() as id");
  idStmt.step();
  const idResult = idStmt.getAsObject();
  idStmt.free();
  return {
    lastInsertRowid: idResult.id,
    changes: 1
  };
}
async function getAllApplications() {
  const database = await initDb();
  const stmt = database.prepare(`
    SELECT * FROM applications
    ORDER BY submittedAt DESC
  `);
  const results = [];
  while (stmt.step()) {
    results.push(stmt.getAsObject());
  }
  stmt.free();
  return results;
}
async function getApplicationById(id) {
  const database = await initDb();
  const stmt = database.prepare("SELECT * FROM applications WHERE id = ?");
  stmt.bind([id]);
  let result = null;
  if (stmt.step()) {
    result = stmt.getAsObject();
  }
  stmt.free();
  return result;
}
async function updateApplicationStatus(id, status, reviewedBy, notes) {
  const database = await initDb();
  const stmt = database.prepare(`
    UPDATE applications
    SET status = ?, reviewedAt = CURRENT_TIMESTAMP, reviewedBy = ?, notes = ?
    WHERE id = ?
  `);
  stmt.bind([status, reviewedBy, notes, id]);
  stmt.step();
  stmt.free();
  saveDb();
  return { changes: 1 };
}
async function verifyAdminLogin(email, password) {
  const database = await initDb();
  const hashedPassword = Buffer.from(password).toString("base64");
  const stmt = database.prepare(
    "SELECT * FROM admin_users WHERE email = ? AND password = ?"
  );
  stmt.bind([email, hashedPassword]);
  let user = null;
  if (stmt.step()) {
    const row = stmt.getAsObject();
    user = { id: row.id, email: row.email, name: row.name };
  }
  stmt.free();
  return user;
}
const handleSubmitApplication = async (req, res) => {
  try {
    const {
      firstName,
      lastName,
      email,
      phone,
      ssn,
      idType,
      address,
      city,
      state,
      zipCode,
      bankName,
      accountType,
      routingNumber,
      accountNumber
    } = req.body;
    if (!firstName || !lastName || !email || !phone || !ssn || !idType || !address || !city || !state || !zipCode || !bankName || !accountType || !routingNumber || !accountNumber) {
      res.status(400).json({ error: "Missing required fields" });
      return;
    }
    const result = await submitApplication({
      firstName,
      lastName,
      email,
      phone,
      ssn,
      idType,
      address,
      city,
      state,
      zipCode,
      bankName,
      accountType,
      routingNumber,
      accountNumber
    });
    res.json({
      success: true,
      id: result.lastInsertRowid,
      message: "Application submitted successfully"
    });
  } catch (error) {
    console.error("Error submitting application:", error);
    res.status(500).json({ error: "Failed to submit application" });
  }
};
const handleAdminLogin = async (req, res) => {
  try {
    const { email, password } = req.body;
    if (!email || !password) {
      res.status(400).json({ error: "Email and password required" });
      return;
    }
    const user = await verifyAdminLogin(email, password);
    if (!user) {
      res.status(401).json({ error: "Invalid email or password" });
      return;
    }
    const token = Buffer.from(JSON.stringify(user)).toString("base64");
    res.status(200).json({
      success: true,
      user: {
        id: user.id,
        email: user.email,
        name: user.name
      },
      token
    });
  } catch (error) {
    console.error("Error logging in:", error);
    res.status(500).json({ error: "Server error during login" });
  }
};
const handleGetApplications = async (req, res) => {
  try {
    const token = req.headers.authorization?.split(" ")[1];
    if (!token) {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    try {
      const user = JSON.parse(Buffer.from(token, "base64").toString());
      if (!user.id) {
        res.status(401).json({ error: "Unauthorized" });
        return;
      }
    } catch {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    const applications = await getAllApplications();
    res.json({ applications });
  } catch (error) {
    console.error("Error fetching applications:", error);
    res.status(500).json({ error: "Failed to fetch applications" });
  }
};
const handleGetApplication = async (req, res) => {
  try {
    const token = req.headers.authorization?.split(" ")[1];
    if (!token) {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    try {
      const user = JSON.parse(Buffer.from(token, "base64").toString());
      if (!user.id) {
        res.status(401).json({ error: "Unauthorized" });
        return;
      }
    } catch {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    const { id } = req.params;
    const application = await getApplicationById(parseInt(id));
    if (!application) {
      res.status(404).json({ error: "Application not found" });
      return;
    }
    res.json({ application });
  } catch (error) {
    console.error("Error fetching application:", error);
    res.status(500).json({ error: "Failed to fetch application" });
  }
};
const handleUpdateApplication = async (req, res) => {
  try {
    const token = req.headers.authorization?.split(" ")[1];
    if (!token) {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    let user;
    try {
      user = JSON.parse(Buffer.from(token, "base64").toString());
      if (!user.id) {
        res.status(401).json({ error: "Unauthorized" });
        return;
      }
    } catch {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    const { id } = req.params;
    const { status, notes } = req.body;
    if (!status) {
      res.status(400).json({ error: "Status required" });
      return;
    }
    await updateApplicationStatus(parseInt(id), status, user.email, notes || "");
    res.json({
      success: true,
      message: "Application updated successfully"
    });
  } catch (error) {
    console.error("Error updating application:", error);
    res.status(500).json({ error: "Failed to update application" });
  }
};
function createServer() {
  const app2 = express__default();
  initializeDatabase();
  app2.use(cors());
  app2.use(express__default.json());
  app2.use(express__default.urlencoded({ extended: true }));
  app2.get("/api/ping", (_req, res) => {
    const ping = process.env.PING_MESSAGE ?? "ping";
    res.json({ message: ping });
  });
  app2.get("/api/demo", handleDemo);
  app2.post("/api/applications/submit", handleSubmitApplication);
  app2.post("/api/admin/login", handleAdminLogin);
  app2.get("/api/applications", handleGetApplications);
  app2.get("/api/applications/:id", handleGetApplication);
  app2.patch("/api/applications/:id", handleUpdateApplication);
  app2.use((err, _req, res, _next) => {
    console.error("Unhandled error:", err);
    res.status(500).json({
      error: "Internal server error",
      message: void 0
    });
  });
  app2.use("/api", (_req, res) => {
    res.status(404).json({ error: "API route not found" });
  });
  return app2;
}
const app = createServer();
const port = process.env.PORT || 3e3;
const __dirname$1 = import.meta.dirname;
const distPath = path.join(__dirname$1, "../spa");
app.use(express.static(distPath));
app.get("*", (req, res) => {
  if (req.path.startsWith("/api/") || req.path.startsWith("/health")) {
    return res.status(404).json({ error: "API endpoint not found" });
  }
  res.sendFile(path.join(distPath, "index.html"));
});
app.listen(port, () => {
  console.log(`🚀 Fusion Starter server running on port ${port}`);
  console.log(`📱 Frontend: http://localhost:${port}`);
  console.log(`🔧 API: http://localhost:${port}/api`);
});
process.on("SIGTERM", () => {
  console.log("🛑 Received SIGTERM, shutting down gracefully");
  process.exit(0);
});
process.on("SIGINT", () => {
  console.log("🛑 Received SIGINT, shutting down gracefully");
  process.exit(0);
});
//# sourceMappingURL=node-build.mjs.map
